<?php
/*******************************************************************************
* File with Clarity page
*******************************************************************************/

function generate_wordpress_id_option_if_empty() {
    $clarity_wp_site = get_option('clarity_wordpress_site_id');
	if (empty($clarity_wp_site)) {
        update_option('clarity_wordpress_site_id', wp_generate_uuid4());
    };
}

/**
* generate a guid identifier for this wordpress site
* runs in the callback of register_activation_hook, rerunning here for existing plugin which updated 
**/
function refresh_wordpress_id_option() {
    update_option('clarity_wordpress_site_id', wp_generate_uuid4());
}

/**
* Displays the embedded iframe in Clarity settings
**/
function clarity_section_iframe_callback() {
    $nonce = wp_create_nonce('wp_ajax_edit_clarity_project_id'); 

    $clarity_project_id_option = get_option(
        'clarity_project_id', /* option */
        clarity_project_id_default_value() /* default */
    );
    $clarity_wp_site = get_option(
        'clarity_wordpress_site_id' /* option */
        /* default */
    );

    $site_url = home_url();

    $clarity_domain = "https://clarity.microsoft.com/embed";

    $query_params = "?nonce=$nonce&integration=Wordpress&wpsite=$clarity_wp_site&siteurl=$site_url";

    // set a QP if user is admin
    if(current_user_can('manage_options')) {
        $query_params = $query_params . "&WPAdmin=1";
    }

    // set a QP if user is WooCommerce plugin is active
    if (class_exists( 'woocommerce' )) { 
        $query_params = $query_params . "&WooCommerce=1"; 
    }

    // initially set iframe src to the new users path
    $iframe_src = $clarity_domain.$query_params;

    // clarity project exist
    if(!empty($clarity_project_id_option))
    {
        $iframe_src = $iframe_src."&project=".$clarity_project_id_option;
    }

	?>
    <div style="width:100%;height:100vh;padding-right:15px;margin-top:0px;box-sizing:border-box;">
        <iframe sandbox="allow-modals allow-forms allow-scripts allow-same-origin allow-popups allow-storage-access-by-user-activation" src="<?php echo $iframe_src ?>" width="100%" height="100%" title="Microsoft Clarity" /> 
    </div>
    <?php
}

/**
* clarity project id default value is empty string
**/
function clarity_project_id_default_value() {
	return '';
}

/**
* Generates a menu page
**/
 
add_action('admin_menu', 'clarity_page_generation');
function clarity_page_generation() {
	add_menu_page(
        'microsoft-clarity', /* $page_title */
        'Clarity', /* menu_title */
        'edit_posts', /* capability */
        'microsoft-clarity', /* menu_slug */
        'clarity_section_iframe_callback', /* callback */
        'https://claritystatic.blob.core.windows.net/images/logo.svg', /* icon_url */
        99 /* position */
    );
}

/**
* Register Plugin settings
* clarity_project_id: option for currently integrated Clarity project id
* clarity_wordpress_site_id: a guid generated by the Clarity plugin to uniquely identify this wordpress site
* clarity_section_iframe_callback: part of the settings page in which the iframe is embedded
* clarity_is_latest_plugin_version: option for checking if the plugin is latest version
* clarity_is_agent_enabled: option for checking if ads agent is connected
**/
add_action('admin_init', 'clarity_register_settings');
function clarity_register_settings() {
	register_setting(
        'clarity_settings_fields', /* $option_group */
        'clarity_project_id' /* option_name */
        /* args */
    );
	register_setting(
        'general', /* $option_group */
        'clarity_wordpress_site_id' /* option_name */
        /* args */
    );
    register_setting(
        'general', /* $option_group */
        'clarity_is_latest_plugin_version' /* option_name */
        /* args */
    );
    register_setting(
        'general', /* $option_group */
        'clarity_is_agent_enabled' /* option_name */
        /* args */
    );
}

/** 
* Notice for when wordpress admins did not finish intalling Clarity
* did not integrate a project
*/
add_action('admin_notices', 'setup_clarity_notice__info');
function setup_clarity_notice__info() {
	global $pagenow;
	$url = get_admin_url() . 'admin.php?page=microsoft-clarity';

    $learnMoreUrl = 'https://wordpress.org/plugins/microsoft-clarity/';

    $clarity_project_id_option = get_option(
        'clarity_project_id', /* option */
        clarity_project_id_default_value() /* default */
    );
    $pageQPExists = isset($_GET['page']);
    if($pageQPExists) {
        $pageQP =  $_GET['page'];
    } else {
        $pageQP = "";
    }


	if(empty($clarity_project_id_option) && $pageQP !== "microsoft-clarity" && current_user_can("manage_options")) {
        echo
        '<div class="notice notice-info is-dismissible">
            <p style="font-weight:700">
                Unlock User Insights with Microsoft Clarity!
            </p>
            <p style="font-weight:500">
                Almost there! Start tracking user behavior on your site with Microsoft Clarity. See exactly where on your site users click, scroll, and get stuck. It takes just a few moments to set up.
            </p>
            <p>
                <a class="button-primary" href="'. $url .'">
                    Setup Clarity
                </a>
                <a class="button-primary" style="margin-left:10px" href="'. $learnMoreUrl .'">
                    Learn more
                </a>
            </p>
        </div>';
    }
}

/**
* Add js function to listen to message on all admin pages
* These message contain changes to integrated Clarity project
* remove - change - add new
*/
add_action('admin_enqueue_scripts', 'add_event_listeners');
function add_event_listeners($hook) {
    $pageQPExists = isset($_GET['page']);
    if($pageQPExists) {
        $pageQP =  $_GET['page'];
    } else {
        $pageQP = "";
    }

    if($pageQP !== "microsoft-clarity") {
        return;
    }

    if(!current_user_can("edit_posts")) {
        return;
    }

    wp_register_script(
        'window_listeners_js', /* handle */
        plugins_url('js\add_window_listeners.js', __FILE__ ), /* src */
        array(), /* deps  */
        false, /* ver  */
        false /* in_footer */
    );
    wp_enqueue_script(
        'window_listeners_js' /* handle */
        /* src */
        /* deps  */
        /* ver  */
        /* in_footer */    
    );
}

/**
* Add callback triggered when a new message is received
* Edits the clarity project id option respectively
*/
add_action('wp_ajax_edit_clarity_project_id', "edit_clarity_project_id");
function edit_clarity_project_id() {
    $new_value = $_POST['new_value'];
    $nonce = $_POST['nonce'];
    if(!wp_verify_nonce($nonce, "wp_ajax_edit_clarity_project_id")) {
        die(
            json_encode(
                array(
                    'success' => false,
                    'message' => 'Invalid nonce.',
                )
            )
        );
    }
    // only admins are allowed to edit the Clarity project id
    if (!current_user_can('manage_options')) {
        die(
            json_encode(
                array(
                    'success' => false,
                    'message' => 'User must be WordPress admin.'
                )
            )
        );
    } else {
        update_option(
            'clarity_project_id', /* option */
            $new_value /* value */
            /* autoload */
        );
        die(
            json_encode(
                array(
                    'success' => true,
                    'message' => 'Clarity project updated successfully.'
                    )
                )
        );
    }
}

/**
* Add callback triggered when a new message is received
* Edits the agent enabled status option respectively
*/
add_action('wp_ajax_edit_agent_enabled_status', "edit_agent_enabled_status");
function edit_agent_enabled_status() {
    $new_value = $_POST['new_value'];
    $nonce = $_POST['nonce'];
    if(!wp_verify_nonce($nonce, "wp_ajax_edit_clarity_project_id")) {
        die(
            json_encode(
                array(
                    'success' => false,
                    'message' => 'Invalid nonce.',
                )
            )
        );
    }
    // only admins are allowed to edit the Clarity project id
    if (!current_user_can('manage_options')) {
        die(
            json_encode(
                array(
                    'success' => false,
                    'message' => 'User must be WordPress admin.'
                )
            )
        );
    } else {
        update_option(
            'clarity_is_agent_enabled', /* option */
            $new_value /* value */
            /* autoload */
        );
        die(
            json_encode(
                array(
                    'success' => true,
                    'message' => 'Agent enabled status updated successfully.'
                    )
                )
        );
    }
}

/**
* Displays an admin notice if the plugin version installed is not the latest
*/
add_action( 'admin_notices', 'plugin_update_notice' );
function plugin_update_notice() {
    // Only show the notice to users who can update plugins
    if ( ! current_user_can( 'update_plugins' ) ) {
        return;
    }

    if ( get_option( 'clarity_is_latest_plugin_version' ) == '1' ) {
        return;
    }

    $plugin_slug = 'microsoft-clarity/clarity.php';
    $update_url = wp_nonce_url(
        add_query_arg(
            array(
                'action'       => 'trigger_plugin_update',
                'plugin'       => urlencode( $plugin_slug ),
            ),
            admin_url( 'admin.php' )
        ),
        'plugin_update_nonce'
    );

    ?>
    <div class="notice notice-warning is-dismissible">
        <p style="font-weight:700">
            <?php _e( 'A new version of Microsoft Clarity is available.', 'text-domain' ); ?>
        </p>
        <p>
            <a href="<?php echo esc_url( $update_url ); ?>" class="button button-primary">
                <?php _e( 'Update Now', 'text-domain' ); ?>
            </a>
        </p>
    </div>
    <?php
}

/**
* Updates the plugin to the latest version programmatically
* The upgrade function deactives the plugin by default before the upgrade, hence the need to reactivate it
*/
add_action( 'admin_action_trigger_plugin_update', 'plugin_perform_update' );
function plugin_perform_update() {
    if ( ! current_user_can( 'update_plugins' ) ) {
        wp_die( __( 'You do not have sufficient permissions to update plugins.', 'text-domain' ) );
    }

    if ( ! isset( $_GET['plugin'] ) || ! isset( $_GET['_wpnonce'] ) ) {
        return;
    }

    $plugin_slug = sanitize_text_field( urldecode( $_GET['plugin'] ) );

    if ( ! wp_verify_nonce( $_GET['_wpnonce'], 'plugin_update_nonce' ) ) {
        wp_die( __( 'Nonce verification failed.', 'text-domain' ) );
    }

    include_once( ABSPATH . 'wp-admin/includes/class-wp-upgrader.php' );
    include_once( ABSPATH . 'wp-admin/includes/plugin.php' );

    // Create a custom skin to handle output and redirection
    $upgrader_skin = new Automatic_Upgrader_Skin();
    $upgrader      = new Plugin_Upgrader( $upgrader_skin );

    // Perform the update
    $updated = $upgrader->upgrade( $plugin_slug );

    if ( is_wp_error( $updated ) || ! $updated ) {
        // Handle error: redirect back to admin page with an error notice 
        $redirect_url = add_query_arg( 'plugin_update_error', '1', admin_url( 'plugins.php' ) );
        wp_redirect( esc_url( $redirect_url ) );
        exit;
    } else {
        // Success: redirect back to the plugin page with a success notice
        activate_plugin( $plugin_slug );
        $redirect_url = add_query_arg( 'plugin_updated', '1', admin_url( 'admin.php?page=microsoft-clarity' ) );
        wp_redirect( esc_url( $redirect_url ) );
        exit;
    }   
}

/**
* Display an admin notice with the status of the plugin update
*/
add_action( 'admin_notices', 'plugin_admin_notices' );
function plugin_admin_notices() {
    if ( isset( $_GET['plugin_updated'] ) && '1' === $_GET['plugin_updated'] ) {
        echo
        '<div class="notice notice-success is-dismissible">
            <p><strong>Microsoft Clarity plugin has been updated successfully.</strong></p>
        </div>'; 
        
    } else if ( isset( $_GET['plugin_update_error'] ) && '1' === $_GET['plugin_update_error'] ) {
         echo
        '<div class="notice notice-error is-dismissible">
            <p><strong>Microsoft Clarity plugin update failed.</strong></p>
        </div>';
    }
}

